/**
 * Andreas Solymosi: Covariance and contravariance in Java
 * Syntax test for Chapter 13. Variance of Lambdas
 * @author  Prof. Solymosi, Beuth University of Applied Sciences, Berlin
 * @version December 6, 2016
 */
package com.solymosi.javaworld.variances.test13;

import java.awt.Button;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Comparator;
import java.util.function.Function;

import com.solymosi.javaworld.variances.SubType;
import com.solymosi.javaworld.variances.SuperType;

interface Type {}

// Syntax test:
@FunctionalInterface
interface Functionalinterface {
	void method(Type parameter);
}
@FunctionalInterface
interface SuperFI {
	void method(SuperType parameter);
}
@FunctionalInterface
interface SubFI {
	void method(SubType parameter);
}
@FunctionalInterface
interface GenericFunctionalInterface<T> {
	void method(T t);
}

@SuppressWarnings("serial")
class SuperException extends Exception {}
@SuppressWarnings("serial")
class SubException extends SuperException {}

@FunctionalInterface
interface SuperInterface {
	SuperType function() throws SuperException;
}
@FunctionalInterface
interface SubInterface {
	SubType function() throws SubException;
}

public class Test13 {
	// Syntax test:
	static //--
	void procedure(Functionalinterface functionalInterface) {
		functionalInterface.method(new Type(){});
	}
	
	static //-- 
	double integral(Function<Double, Double> function, double a, double b) {
		return 0;
	}
	
	static //-- 
	void covariantProcedure(SuperFI fi, SuperType superParameter) {
		fi.method(superParameter);
	}
	static //.. 
	void contravariantProcedure(SubFI sfi, SuperType subParameter) {
		sfi.method((SubType)subParameter);
	}
	static //.. 
	SuperType superFunction(SuperInterface superParameter) throws SuperException {
		return superParameter.function(); // or do something more interesting
	}
	static //.. 
	SubType subFunction(SubInterface subParameter) throws SubException {
		return subParameter.function(); // similarly
	}

	@SuppressWarnings("unused")
	public static void main(String[] args) throws SuperException {
		procedure(parameter -> { //  // lambda
			System.out.println("Lambda test: " + parameter); //--
		});
		// anonymous method passed as parameter to procedure
		
		procedure(new Functionalinterface() {
			public void method(Type parameter) {
				System.out.println("Lambda test: " + parameter); //--
			}
		});
		
		class Implementation implements Functionalinterface {
			public void method(Type parameter) { } }
		
		Functionalinterface fi = new Implementation();
		procedure(fi);

		procedure(new Functionalinterface() {
			public void method(Type parameter) { } } );

		double result = integral((Double x) -> Math.sin(x), 0, Math.PI);
				
		Button button = new Button(); //--
		button.addActionListener( // pre-Java 8 version
				new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						System.out.println("Button pushed"); }});
		
		button.addActionListener(
		  e -> System.out.println("Button pushed") // lambda expression
		);
		button.addActionListener(System.out::println);
		
		ActionListener actionListener = e -> System.out.println("Button pushed");
		button.addActionListener(actionListener);
		
		class OldRunnable implements Runnable {
			  public void run() {
			    System.out.println("Old");
			  }
			};
		Runnable old = new OldRunnable();
		new Thread(old).start();

		new Thread(() -> System.out.println("New")).start(); 

		Comparator<String> c;
		c = (String left, String right) -> left.compareTo(right);
		c = (left, right) -> left.compareTo(right); // equivalent

		System.out.println(c.compare("Hallo", "World"));
		
		covariantProcedure(parameter -> {}, new SubType());	 // SubType is compatible to SuperType
		// contravariantProcedure(parameter -> {}, new SuperType()); // error: ClassCastException

		SuperType superVariable = superFunction(() -> new SuperType()); // 000 normal
		superVariable = subFunction(() -> new SubType()); // 011 simply compatible
		superVariable = superFunction(() -> new SubType()); // 001 covariant
		// superVariable = subFunction(() -> new SuperType()); // 010 not applicable

		SubType subVariable = (SubType) superFunction(() -> new SuperType()); // 100 ClassCastException
		// superVariable = subFunction(() -> new SuperType()); // 101 not applicable
		// subVariable = subFunction(() -> new SuperType()); // 110 cannot convert from SuperType to SubType		
		subVariable = (SubType) superFunction(() -> new SubType()); // 111 normal			

		GenericFunctionalInterface<?> wildcardReference;
		wildcardReference = (String string) -> System.out.println(string);
		// wildcardReference.method("Wildcard"); // not applicable
		wildcardReference = (Integer integer) -> integer++;
		// wildcardReference.method(1); // not applicable
		// wildcardReference.method(new Object()); // not applicable

		GenericFunctionalInterface<? extends Type> covariantReference;
		covariantReference = (Type parameter) -> System.out.println(parameter);
		// covariantReference.method(new Type(){}); // not applicable
		
		GenericFunctionalInterface<? super SubType> contravariantReference;
		contravariantReference = (SuperType parameter) -> System.out.println(parameter);
		contravariantReference.method(new SubType(){});
		// contravariantReference.method(new SuperType(){});
	}
}
